% This code estimates the generalized bvp model in Han and Lee (2019)

% Semiparametric Model : Transformed model 
% Choice of G: Standard normal
% Parametric Specification : Normal 
% True Marginal Distribution : Standard Normal 


clear;
clc;

%% Design of Simulation : Parameters

n = 500 ; % Number of observations
m = 2000; % Number of iterations

kne = floor(3*(n^(1/7))); % Number of Polynomials for epsilon 
knv = floor(3*(n^(1/7)));

cop = [0;2; 3; 5];

for cop_ind = 1:4

cop_opt = cop(cop_ind,1) ;    % Gaussian = 0, Frank = 2, Clayton = 3, Gumbel = 5
cop_opt_DGP = cop(cop_ind,1);

beta1 = -1 ;
beta2 = 0.8  ;
%beta = [beta1; beta2]; 
beta = beta1 ;

dx = size(beta,1) ;

alpha1 = -1 ;
alpha2 = 0.5;
%alpha = [alpha1; alpha2];
alpha = alpha1 ;

delta = 1.1;
gamma = 0.8 ;

dz = size(gamma,1) ;

beta_base = beta(1) ;
alpha_base = alpha(1) ;

%% DGP : Marginal Distributions and Covariates

mu_xz = zeros(dx+dz,1);
mu_x = mu_xz(1:dx,1) ;
mu_z = mu_xz(dx+1:dx+dz) ;

corr12 = -0.2 ;
corr1z = -0.1 ;
corr2z = 0.1 ;

if dx == 1 
    sigma_xz = [1, corr1z; corr1z, 1];
    
elseif dx == 2
    sigma_xz = [1, corr12, corr1z ; corr12, 1, corr2z; corr1z, corr2z, 1];
    
end

 
mu_e = 0  ;
mu_v = 0 ;

sigma_e = sqrt(1) ;
sigma_v = sqrt(1) ;

nuisance = [mu_e; mu_v; sigma_e; sigma_v] ;

ate = normcdf(mu_x'*beta + delta, mu_e, sigma_e) - normcdf(mu_x'*beta, mu_e, sigma_e) ;

%% DGPs : Copula


% Normal copula
if cop_opt_DGP == 0
    rho = 0.5176;  % sp_rho = 0.5, NO LARGER THAN 0.9!!
    sp_rho = copulastat('Gaussian',rho,'type','Spearman') ;
end
% Example 3.2: The Frank family
if cop_opt_DGP == 2
    rho = 3.446;    % sp_rho = 0.5  
    sp_rho = copulastat('Frank',rho,'type','Spearman');
end    
% Example 3.3: The Clayton family (The Kimeldorf and Sampson family, in Joe)
if cop_opt_DGP == 3
    rho = 1.076;    % sp_rho = 0.5
    sp_rho = copulastat('Clayton',rho,'type','Spearman');
end
% Example 3.5: The Gumbel family
if cop_opt_DGP == 5
    rho = 1.541;    % sp_rho = 0.5     
    sp_rho = copulastat('Gumbel',rho,'type','Spearman');
end

%% Specifications : Copula

% Normal copula
if cop_opt == 0
    rho_mid = 0.5 ;
    rho_L = -0.9;
    rho_H = 0.9;
end
% Example 3.2: The Frank family
if cop_opt == 2
    rho_mid = 3.446;    % sp_rho = 0.5
    rho_L = rho_mid - 2;   % lower bound is infinity
    rho_H = rho_mid + 2;    % upper bound is infinity
    %EXCEPT rho = 0;
end    
% Example 3.3: The Clayton family (The Kimeldorf and Sampson family, in Joe)
if cop_opt == 3
    rho_mid = 1.076;    % sp_rho = 0.5
    rho_L = 0 + 0.01;   % lower bound is -1, but somehow error occurs...
    rho_H = rho_mid + 2;    % upper bound is infinity
    %EXCEPT rho = 0;
end% Example 3.5: The Gumbel family
if cop_opt == 5
    rho_mid = 1.541;    % sp_rho = 0.5
    rho_L = 1 + 0.01;   % lower bound is 1
    rho_H = rho_mid + 2;    % upper bound is infinity
end



%% Option 
options = optimset('LargeScale','off', ...
               'HessUpdate','bfgs', ...
               'Algorithm', 'active-set',...
               'Hessian','off', ...
               'GradObj','off', ...
               'DerivativeCheck','off',...
               'Display', 'off');



%% Matrice for Results 

theta = [alpha; gamma; beta; delta; rho] ;

kk = size(theta,1) ; % Number of finite-dimensional parameters

theta_ext = [theta; sp_rho] ;

semi_theta_est_all = zeros(m,kk) ;
xie_est_all = zeros(m,kne) ;
conse_est_all = zeros(m,1) ;
xiv_est_all = zeros(m,knv) ;
consv_est_all = zeros(m,1) ;

semi_sp_rho_est_all = zeros(m,1);

semi_ate_est_all = zeros(m,1);


norm_theta_est_all = zeros(m,kk) ;
norm_sp_rho_est_all = zeros(m,1) ;

norm_mue_est_all = zeros(m,1) ;
norm_muv_est_all = zeros(m,1) ;
norm_sigmae_est_all = zeros(m,1) ;
norm_sigmav_est_all = zeros(m,1) ;

norm_ate_est_all = zeros(m,1) ;

%% Estimation

if cop_opt == 0
    copula_hat = @(u1,u2,rho)copulacdf('Gaussian',[u1,u2],[1,rho;rho,1]);
    %copula = @(u1,u2,rho)normcop(u1,u2,rho);
elseif cop_opt == 2
    copula_hat = @(u1,u2,rho)copulacdf('Frank',[u1,u2],rho);
elseif cop_opt == 3
    copula_hat = @(u1,u2,rho)copulacdf('Clayton',[u1,u2],rho);
elseif cop_opt == 5
    copula_hat = @(u1,u2,rho)copulacdf('Gumbel',[u1,u2],rho);
end;


parfor j=1:m 
    %Initial values 
    theta_0 = theta + ((-1 + 2*rand(kk,1))/10) ; 
    theta_0(1) = alpha_base ;
    theta_0(dx+dz+1) = beta_base;
    theta_0(end) = rho_mid ;
    
    xie_0 = zeros(kne,1) ;
    xiv_0 = zeros(knv,1) ;
    cons_0 = 1 ;
    
    semi_par_0 = [theta_0 ; xie_0; cons_0; xiv_0; cons_0] ;
    
    mu_0 = 0 ;
    sigma_0 = 1 ;
    
    norm_par_0 = [theta_0 ; mu_0; mu_0; sigma_0; sigma_0] ;
    
    if cop_opt_DGP == 0
        sigma_u =[1,rho;
            rho,1];
        u = copularnd('Gaussian',sigma_u,n);
    end
    
    if cop_opt_DGP == 2
        u = copularnd('Frank',rho,n);
        %u = copula2rnd(n, rho);
    end
    
    if cop_opt_DGP == 3
        u = copularnd('Clayton',rho,n);
    end
    
    if cop_opt_DGP == 5
        u = copularnd('Gumbel',rho,n);
        %u = copula5rnd(n, rho);
    end
    
    u1 = u(:,1);
    u2 = u(:,2);
    
    %x1 = normrnd(0,1,n,1)*sigma_x1 + mu_x1 ;
    %x2 = normrnd(0,1,n,1)*sigma_x2 + mu_x2 ;
    
    xz = mvnrnd(mu_xz, sigma_xz, n) ;
    
    x = xz(:,1:dx);
    z = xz(:,(dx+1):(dx+dz)) ;
    
    x_bar = mean(x) ;
    
    %if dx == 1 
    %    x = x1 ;
    %else
    %    x = [x1, x2];
    %end
       
    %z = normrnd(0,1,n,1)*sigma_z + mu_z ;
        
    d = (normcdf((x*alpha + z*gamma - mu_v)./sigma_v, 0, 1) >= u1);  
    y = (normcdf((x*beta + delta*d - mu_e)./sigma_e, 0, 1) >= u2);   
    
    data = [y,d,x,z];
    
    [semi_par_est, semi_logL_val] = fmincon(@(par)semi_logl(data, par, cop_opt, dx, dz, kk, kne, knv),semi_par_0, [], [], [], [], [], [], @(par)semi_constraint(par, dx, dz, kk, kne, knv, beta_base, alpha_base, rho_H, rho_L), options);
    
    semi_theta_est_all(j,:) = semi_par_est(1:kk)' ;
    
    semi_beta_est = semi_par_est(dx+dz+1:dx+dz+dx);
    semi_delta_est = semi_par_est(dx+dz+dx+1) ;
        
    xie_est_all(j,:) = semi_par_est(kk+1:kk+kne)' ;
    conse_est_all(j,:) = semi_par_est(kk+kne+1) ;
    xiv_est_all(j,:) = semi_par_est(kk+kne+2:kk+kne+1+knv) ;
    consv_est_all(j,:) = semi_par_est(kk+kne+knv+2) ;
    
    semi_ate_est_all(j,1) = dist(normcdf(x_bar*semi_beta_est + semi_delta_est,0,1), semi_par_est(kk+1:kk+kne+1), kne) - dist(normcdf(x_bar*semi_beta_est,0,1), semi_par_est(kk+1:kk+kne+1), kne);
    
    
    [norm_par_est, norm_logL_val] = fmincon(@(par)norm_logl(data, par, cop_opt, dx, dz, kk),norm_par_0, [], [], [], [], [], [], @(par)norm_constraint(par, dx, dz, kk, beta_base, alpha_base, rho_H, rho_L) , options);
    
    norm_beta_est = norm_par_est(dx+dz+1:dx+dz+dx);
    norm_delta_est = norm_par_est(dx+dz+dx+1) ;
    
    norm_theta_est_all(j,:) = norm_par_est(1:kk)';
    norm_mue_est_all(j,1) = norm_par_est(kk+1) ;
    norm_muv_est_all(j,1) = norm_par_est(kk+2) ;
    norm_sigmae_est_all(j,1) = norm_par_est(kk+3) ;
    norm_sigmav_est_all(j,1) = norm_par_est(kk+4) ;        
    
    norm_ate_est_all(j,1) = normcdf(x_bar*norm_beta_est + norm_delta_est, norm_mue_est_all(j,1), norm_sigmae_est_all(j,1)) - normcdf(x_bar*norm_beta_est, norm_mue_est_all(j,1), norm_sigmae_est_all(j,1)); 
          
    if cop_opt == 0 
        semi_sp_rho_est_all(j,1) = copulastat('Gaussian',semi_par_est(kk),'type','Spearman');
        norm_sp_rho_est_all(j,1) = copulastat('Gaussian',norm_par_est(kk),'type','Spearman');
        
    elseif cop_opt == 2 
        semi_sp_rho_est_all(j,1) = copulastat('Frank',semi_par_est(kk),'type','Spearman');
        norm_sp_rho_est_all(j,1) = copulastat('Frank',norm_par_est(kk),'type','Spearman');
    elseif cop_opt == 3 
        semi_sp_rho_est_all(j,1) = copulastat('Clayton',semi_par_est(kk),'type','Spearman');
        norm_sp_rho_est_all(j,1) = copulastat('Clayton',norm_par_est(kk),'type','Spearman');
        
    elseif cop_opt == 5
        semi_sp_rho_est_all(j,1) = copulastat('Gumbel',semi_par_est(kk),'type','Spearman');
        norm_sp_rho_est_all(j,1) = copulastat('Gumbel',norm_par_est(kk),'type','Spearman');
    end
    
end;

norm_nuisance_est_all = [norm_mue_est_all, norm_muv_est_all, norm_sigmae_est_all, norm_sigmav_est_all];


%% Performance

semi_theta_ext_est_all = [semi_theta_est_all, semi_sp_rho_est_all] ;
norm_theta_ext_est_all = [norm_theta_est_all, norm_sp_rho_est_all] ;

semi_mean_theta_ext = mean(semi_theta_ext_est_all,1);
norm_mean_theta_ext = mean(norm_theta_ext_est_all,1) ;
norm_mean_nuisance = mean(norm_nuisance_est_all,1) ;

semi_mean_ate = mean(semi_ate_est_all,1) ;
norm_mean_ate = mean(norm_ate_est_all,1) ;

semi_bias = semi_mean_theta_ext - theta_ext';
semi_bias_sq = semi_bias.^2 ;

semi_ate_bias = semi_mean_ate - ate ;
semi_ate_bias_sq = semi_ate_bias.^2 ;

norm_bias = norm_mean_theta_ext - theta_ext' ;
norm_bias_sq = norm_bias.^2 ;

norm_ate_bias = norm_mean_ate - ate ;
norm_ate_bias_sq = norm_ate_bias.^2 ;


norm_nuis_bias = norm_mean_nuisance - nuisance' ;
norm_nuis_bias_sq = norm_nuis_bias.^2 ;

semi_demean_sq = (semi_theta_ext_est_all - kron(semi_mean_theta_ext,ones(m,1))).^2;
semi_var = mean(semi_demean_sq,1);
semi_sd = sqrt(semi_var);

semi_ate_demean_sq = (semi_ate_est_all - kron(semi_mean_ate,ones(m,1))).^2;
semi_ate_var = mean(semi_ate_demean_sq,1);
semi_ate_sd = sqrt(semi_ate_var);

norm_demean_sq = (norm_theta_ext_est_all - kron(norm_mean_theta_ext,ones(m,1))).^2;
norm_var = mean(norm_demean_sq,1);
norm_sd = sqrt(norm_var);

norm_nuis_demean_sq = (norm_nuisance_est_all - kron(norm_mean_nuisance,ones(m,1))).^2;
norm_nuis_var = mean(norm_nuis_demean_sq,1);
norm_nuis_sd = sqrt(norm_nuis_var);

norm_ate_demean_sq = (norm_ate_est_all - kron(norm_mean_ate,ones(m,1))).^2;
norm_ate_var = mean(norm_ate_demean_sq,1);
norm_ate_sd = sqrt(norm_ate_var);

semi_MSE = semi_bias_sq + semi_var;
norm_MSE = norm_bias_sq + norm_var ;

norm_nuis_MSE = norm_nuis_bias_sq + norm_nuis_var ;

semi_ate_MSE = semi_ate_bias_sq + semi_ate_var ;
norm_ate_MSE = norm_ate_bias_sq + norm_ate_var ;


norm_nuis_TABLE = [nuisance'; norm_mean_nuisance; norm_nuis_sd; norm_nuis_bias; norm_nuis_var; norm_nuis_bias_sq; norm_nuis_MSE];


semi_TABLE = [theta_ext'; semi_mean_theta_ext; semi_sd; semi_bias; semi_var; semi_bias_sq; semi_MSE]; 
norm_TABLE = [theta_ext'; norm_mean_theta_ext; norm_sd; norm_bias; norm_var; norm_bias_sq; norm_MSE]; 

par_TABLE = [norm_TABLE, norm_nuis_TABLE] ;

semi_ate_TABLE = [ate; semi_mean_ate; semi_ate_sd; semi_ate_bias; semi_ate_MSE];
norm_ate_TABLE = [ate; norm_mean_ate; norm_ate_sd; norm_ate_bias; norm_ate_MSE];
ate_TABLE = [semi_ate_TABLE, norm_ate_TABLE] ;


TABLE = [semi_TABLE, par_TABLE];


save(sprintf('Correct_specification_%d_%d_%d_%d_k%d.mat',n,m,cop_opt_DGP,cop_opt,kne),'semi_theta_ext_est_all', 'norm_theta_ext_est_all', 'norm_nuisance_est_all', 'TABLE','semi_TABLE', 'par_TABLE', 'cop_opt_DGP','cop_opt', 'xie_est_all', 'xiv_est_all', 'conse_est_all', 'consv_est_all', 'ate_TABLE', 'semi_ate_est_all', 'norm_ate_est_all');

end
